#!/usr/bin/env python3
import csv
import random
from datetime import date, timedelta
from pathlib import Path

# --- Config ---
OUTPUT_PATH = Path("demo_invoices.csv")
NUM_ROWS = 1000
AMOUNT_MIN = 100.00
AMOUNT_MAX = 10000.00
BASE_DATE = date(2024, 1, 1)  # add 1..365 days

# 10 fictive customers with IDs 1000..1009
customers = [
    {"id": 1000, "name": "Rachel Green"},
    {"id": 1001, "name": "Ross Geller"},
    {"id": 1002, "name": "Monica Geller"},
    {"id": 1003, "name": "Chandler Bing"},
    {"id": 1004, "name": "Joey Tribbiani"},
    {"id": 1005, "name": "Phoebe Buffay"},
    {"id": 1006, "name": "Jim Halpert"},
    {"id": 1007, "name": "Pam Beesly"},
    {"id": 1008, "name": "Michael Scott"},
    {"id": 1009, "name": "Dwight Schrute"},
]

# 5 categories, two customers in each
categories = ["CategoryA", "CategoryB", "CategoryC", "CategoryD", "CategoryE"]
for idx, cust in enumerate(customers):
    cust["category"] = categories[idx // 2]

# 4 sellers (film characters)
sellers = [
    "Tony Stark",
    "Natasha Romanoff",
    "Ethan Hunt",
    "Lara Croft",
]

# 6 product categories (furniture)
products = [
    "Sofa",
    "Armchair",
    "Dining Table",
    "Office Chair",
    "Bed Frame",
    "Bookshelf",
]

# Optional: set a seed for reproducibility (uncomment to fix results)
# random.seed(42)

def random_invoice_date():
    # days 1..365 from Jan 1, 2024 → Jan 2..Dec 31, 2024
    delta_days = random.randint(1, 365)
    return (BASE_DATE + timedelta(days=delta_days)).isoformat()

def random_amount():
    return f"{round(random.uniform(AMOUNT_MIN, AMOUNT_MAX), 2):.2f}"

def main():
    with OUTPUT_PATH.open("w", newline="", encoding="utf-8") as f:
        writer = csv.writer(f)
        writer.writerow([
            "CustomerID",
            "CustomerName",
            "Category",
            "Seller",
            "Product",
            "InvoiceDate",
            "InvoiceAmount",
        ])
        for _ in range(NUM_ROWS):
            cust = random.choice(customers)
            writer.writerow([
                cust["id"],
                cust["name"],
                cust["category"],
                random.choice(sellers),
                random.choice(products),
                random_invoice_date(),
                random_amount(),
            ])
    print(f"Wrote {NUM_ROWS} rows to {OUTPUT_PATH.resolve()}")

if __name__ == "__main__":
    main()
    